{ config, lib, pkgs, ... }:
let
  cfg = config.virtualisation.vmware.guest;
  open-vm-tools = if cfg.headless then pkgs.open-vm-tools-headless else pkgs.open-vm-tools;
  xf86inputvmmouse = pkgs.xorg.xf86inputvmmouse;
in
{
  imports = [
    (lib.mkRenamedOptionModule [ "services" "vmwareGuest" ] [ "virtualisation" "vmware" "guest" ])
  ];

  options.virtualisation.vmware.guest = {
    enable = lib.mkEnableOption "VMWare Guest Support";
    headless = lib.mkOption {
      type = lib.types.bool;
      default = !config.services.xserver.enable;
      defaultText = "!config.services.xserver.enable";
      description = "Whether to disable X11-related features.";
    };
  };

  config = lib.mkIf cfg.enable {
    assertions = [ {
      assertion = pkgs.stdenv.hostPlatform.isx86 || pkgs.stdenv.hostPlatform.isAarch64;
      message = "VMWare guest is not currently supported on ${pkgs.stdenv.hostPlatform.system}";
    } ];

    boot.initrd.availableKernelModules = [ "mptspi" ];
    boot.initrd.kernelModules = lib.optionals pkgs.stdenv.hostPlatform.isx86 [ "vmw_pvscsi" ];

    environment.systemPackages = [ open-vm-tools ];

    systemd.services.vmware =
      { description = "VMWare Guest Service";
        wantedBy = [ "multi-user.target" ];
        after = [ "display-manager.service" ];
        unitConfig.ConditionVirtualization = "vmware";
        serviceConfig.ExecStart = "${open-vm-tools}/bin/vmtoolsd";
      };

    # Mount the vmblock for drag-and-drop and copy-and-paste.
    systemd.mounts = lib.mkIf (!cfg.headless) [
      {
        description = "VMware vmblock fuse mount";
        documentation = [ "https://github.com/vmware/open-vm-tools/blob/master/open-vm-tools/vmblock-fuse/design.txt" ];
        unitConfig.ConditionVirtualization = "vmware";
        what = "${open-vm-tools}/bin/vmware-vmblock-fuse";
        where = "/run/vmblock-fuse";
        type = "fuse";
        options = "subtype=vmware-vmblock,default_permissions,allow_other";
        wantedBy = [ "multi-user.target" ];
      }
    ];

    security.wrappers.vmware-user-suid-wrapper = lib.mkIf (!cfg.headless) {
        setuid = true;
        owner = "root";
        group = "root";
        source = "${open-vm-tools}/bin/vmware-user-suid-wrapper";
      };

    environment.etc.vmware-tools.source = "${open-vm-tools}/etc/vmware-tools/*";

    services.xserver = lib.mkIf (!cfg.headless) {
      modules = lib.optionals pkgs.stdenv.hostPlatform.isx86 [ xf86inputvmmouse ];

      config = lib.optionalString (pkgs.stdenv.hostPlatform.isx86) ''
          Section "InputClass"
            Identifier "VMMouse"
            MatchDevicePath "/dev/input/event*"
            MatchProduct "ImPS/2 Generic Wheel Mouse"
            Driver "vmmouse"
          EndSection
        '';

      displayManager.sessionCommands = ''
          ${open-vm-tools}/bin/vmware-user-suid-wrapper
        '';
    };

    services.udev.packages = [ open-vm-tools ];
  };
}
