{ lib, stdenv, fetchFromGitHub
, autoreconfHook, pkg-config
, glibcLocales, kmod, coreutils, perl
, dmidecode, hwdata, sqlite, libtraceevent
, fetchpatch
, nixosTests
}:

stdenv.mkDerivation rec {
  pname = "rasdaemon";
  version = "0.8.0";

  src = fetchFromGitHub {
    owner = "mchehab";
    repo = "rasdaemon";
    rev = "v${version}";
    sha256 = "sha256-BX3kc629FOh5cnD6Sa/69wKdhmhT3Rpz5ZvhnD4MclQ=";
  };

  patches = [
    (fetchpatch { # fix #295002 (segfault on AMD), will be in the release after 0.8.0
      name = "fix crash on AMD";
      url = "https://github.com/mchehab/rasdaemon/commit/f1ea76375281001cdf4a048c1a4a24d86c6fbe48.patch";
      hash = "sha256-1VPDTrAsvZGiGbh52EUdG6tYV/n6wUS0mphOSXzran0=";
    })
  ];

  nativeBuildInputs = [ autoreconfHook pkg-config ];

  buildInputs = [
    coreutils
    glibcLocales
    hwdata
    kmod
    sqlite
    libtraceevent
    (perl.withPackages (ps: with ps; [ DBI DBDSQLite ]))
  ]
  ++ lib.optionals (!stdenv.hostPlatform.isAarch64) [ dmidecode ];

  configureFlags = [
    "--sysconfdir=/etc"
    "--localstatedir=/var"
    "--with-sysconfdefdir=${placeholder "out"}/etc/sysconfig"
    "--enable-all"
  ];

  # The installation attempts to create the following directories:
  # /var/lib/rasdaemon
  #   location of the RAS event log generated by rasdaemon -r
  # /etc/ras/dimm_labels.d
  #   location of the DIMM labels generated by ras-mc-ctl
  # /etc/sysconfig/rasdaemon
  #   location of rasdaemon config file, currently only used for CE PFA config

  # these are optional (for logging, DIMM label storage and user config)
  # /var/lib/rasdaemon should be created by the NixOS module
  # /etc/ras/dimm_labels.d should probably be generated,
  # from user supplied content, in the NixOS module
  # /etc/sysconfig/rasdaemon should be generated if there is user supplied content
  # and default to $out/etc/sysconfig/rasdaemon which should hold the supplied default

  # therefore, stripping these from the generated Makefile
  # (needed in the config flags because those set where the tools look for these)

# easy way out, ends up installing /nix/store/...rasdaemon/bin in $out

  postConfigure = ''
    substituteInPlace Makefile \
      --replace '"$(DESTDIR)/etc/ras/dimm_labels.d"' '"$(prefix)/etc/ras/dimm_labels.d"'
  '';

  outputs = [ "out" "dev" "man" "inject" ];

  postInstall = ''
    install -Dm 0755 contrib/edac-fake-inject $inject/bin/edac-fake-inject
    install -Dm 0755 contrib/edac-tests $inject/bin/edac-tests
  '';

  postFixup = ''
    # Fix dmidecode and modprobe paths
    substituteInPlace $out/bin/ras-mc-ctl \
      --replace 'find_prog ("modprobe")  or exit (1)' '"${kmod}/bin/modprobe"'
  ''
  + lib.optionalString (!stdenv.hostPlatform.isAarch64) ''
    substituteInPlace $out/bin/ras-mc-ctl \
      --replace 'find_prog ("dmidecode")' '"${dmidecode}/bin/dmidecode"'
  '';

  passthru.tests = nixosTests.rasdaemon;

  meta = with lib; {
    description = ''
      A Reliability, Availability and Serviceability (RAS) logging tool using EDAC kernel tracing events
    '';
    longDescription = ''
      Rasdaemon is a RAS (Reliability, Availability and Serviceability) logging
      tool. It records memory errors, using the EDAC tracing events. EDAC is a
      Linux kernel subsystem with handles detection of ECC errors from memory
      controllers for most chipsets on i386 and x86_64 architectures. EDAC
      drivers for other architectures like arm also exists.
    '';
    homepage = "https://github.com/mchehab/rasdaemon";
    license = licenses.gpl2Plus;
    platforms = platforms.linux;
    changelog = "https://github.com/mchehab/rasdaemon/blob/v${version}/ChangeLog";
    maintainers = with maintainers; [ evils ];
  };
}
